package gov.nasa.worldwind.formats.rpf;

import gov.nasa.worldwind.formats.nitfs.*;
import gov.nasa.worldwind.*;

import java.io.*;
import java.text.*;
import java.awt.image.*;

import java.awt.*;
import java.awt.image.*;
import java.io.*;
import java.net.*;
import java.nio.*;
import javax.imageio.*;
import javax.swing.*;
/*
Copyright (C) 2001, 2007 United States Government
as represented by the Administrator of the
National Aeronautics and Space Administration.
All Rights Reserved.
*/

/**
 * @author lado
 * @version $Id: RpfImageFile Apr 8, 2007 8:51:57 AM
 */
public class RpfImageFile extends RpfFile
{
    private NitfsImageSegment imageSegment = null;
    private UserDefinedImageSubheader imageSubheader = null;
    private RpfFrameFileComponents rpfFrameFileComponents = null;

    public RpfFrameFileComponents getRpfFrameFileComponents()
    {
        return this.rpfFrameFileComponents;
    }

    public UserDefinedImageSubheader getImageSubheader()
    {
        return this.imageSubheader;
    }

    public NitfsImageSegment getImageSegment()
    {
        return this.imageSegment;
    }


    
    private RpfImageFile(java.io.File rpfFile) throws java.io.IOException, NitfsRuntimeException
    {
        super(rpfFile);

        this.imageSegment = (NitfsImageSegment) this.getNitfsSegment(NitfsSegmentType.ImageSegment);
        this.validateRpfImage();

        this.imageSubheader = this.imageSegment.getUserDefinedImageSubheader();
        this.rpfFrameFileComponents = this.imageSubheader.getRpfFrameFileComponents();
    }

    private void validateRpfImage() throws NitfsRuntimeException
    {
        if ( null == this.imageSegment )
            throw new NitfsRuntimeException("NitfsReader.ImageSegmentWasNotFound");
        if( null == this.imageSegment.getUserDefinedImageSubheader())
            throw new NitfsRuntimeException("NitfsReader.UserDefinedImageSubheaderWasNotFound");
        if( null == this.imageSegment.getUserDefinedImageSubheader().getRpfFrameFileComponents())
            throw new NitfsRuntimeException("NitfsReader.RpfFrameFileComponentsWereNotFoundInUserDefinedImageSubheader");
    }

    public int[] getImagePixelsAsArray(int[] dest, RpfImageType imageType)
    {
        IntBuffer buffer = IntBuffer.wrap(dest);
        this.getImagePixelsAsBuffer(buffer, imageType);
        return dest;
    }

    public ByteBuffer getImageAsDdsTexture()
    {
        if (null != this.imageSegment)
            return this.imageSegment.getImageAsDdsTexture();
        return null;
    }


    public IntBuffer getImagePixelsAsBuffer(IntBuffer dest, RpfImageType imageType)
    {
        if (null != this.imageSegment)
            this.imageSegment.getImagePixelsAsArray(dest, imageType);
        return dest;
    }

    public BufferedImage getBufferedImage()
    {
        if (null == this.imageSegment)
            return null;

        BufferedImage bimage = new BufferedImage(
            this.getImageSegment().numSignificantCols,
            this.getImageSegment().numSignificantRows,
            BufferedImage.TYPE_INT_ARGB);

        WritableRaster raster = bimage.getRaster();
        java.awt.image.DataBufferInt dataBuffer = (java.awt.image.DataBufferInt) raster.getDataBuffer();

        IntBuffer buffer = IntBuffer.wrap(dataBuffer.getData());
        this.getImageSegment().getImagePixelsAsArray(buffer, RpfImageType.IMAGE_TYPE_ALPHA_RGB);
        return bimage;
    }

    public boolean hasTransparentAreas()
    {
        if(null != this.imageSegment)
            return (this.imageSegment.hasTransparentPixels() || this.imageSegment.hasMaskedSubframes());
        return false;
    }

    public static RpfImageFile load(java.io.File rpfFile) throws java.io.IOException, NitfsRuntimeException
    {
        return new RpfImageFile(rpfFile);
    }

    public static void main(String args[]) throws IOException
    {
        String cibFilename = (Configuration.isWindowsOS())
            ? "C:\\depot\\WorldWindJ\\utils\\gdal\\020g222a.i42" : "/depot/WorldWindJ/utils/gdal/020g222a.i42";
        String cadrgFilename = (Configuration.isWindowsOS())
                    ? "C:\\depot\\nitfs\\CADRG\\CTLM50\\CT50Z02\\02F7W053.TL2" : "/depot/nitfs/CADRG/CTLM50/CT50Z02/02F7W053.TL2";
        String transparentCadrgFilename = (Configuration.isWindowsOS())
                    ? "C:\\depot\\nitfs\\CADRG\\CTLM50\\CT50Z02\\0D6MM013.TL1" : "/depot/nitfs/CADRG/CTLM50/CT50Z02/0D6MM013.TL1";

        String rpfFilename = cibFilename; // cibFilename; // cadrgFilename; transparentCadrgFilename;

        try
        {
            long startTime = System.currentTimeMillis();

            RpfImageFile rpfImageFile = RpfImageFile.load(new File(rpfFilename));

// ---------- getImageAsDdsTexture example -----------------------

            ByteBuffer ddsBuffer = rpfImageFile.getImageAsDdsTexture();
            System.out.println(MessageFormat.format("RPF file loaded in {0} mSec", (System.currentTimeMillis() - startTime)));
            WWIO.saveBuffer(ddsBuffer, new File(
                (Configuration.isWindowsOS()) ?  "c:\\depot\\nitfs\\DDS\\test.dds" : "/depot/nitfs/DDS/test.dds"
            ));

// ---------- getImageAsArray example -----------------------
//
//            int size = rpfImageFile.getImageSegment().numSignificantCols
//                * rpfImageFile.getImageSegment().numSignificantRows;
//            IntBuffer rgbaBuffer = IntBuffer.allocate(size);
//            rpfImageFile.getImagePixelsAsArray(rgbaBuffer.array(), RpfImageType.IMAGE_TYPE_RGB_ALPHA);
//
//            System.out.println(MessageFormat.format("RPF file loaded in {0} mSec",
//                (System.currentTimeMillis() - startTime)));

// ---------- getBufferedImage example -----------------------
//            BufferedImage bimage = rpfImageFile.getBufferedImage();
//
//            System.out.println(MessageFormat.format("RPF file loaded in {0} mSec",
//                (System.currentTimeMillis() - startTime)));
//
//            Icon icon = new ImageIcon(bimage);
//            JLabel label = new JLabel(icon);
//
//            final JFrame f = new JFrame("RPF Viewer");
//            f.setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE);
//            f.getContentPane().add(label);
//            f.pack();
//            SwingUtilities.invokeLater(new Runnable()
//            {
//                public void run()
//                {
//                    f.setLocationRelativeTo(null);
//                    f.setVisible(true);
//                }
//            });
        }
        catch (Exception e)
        {
            System.out.println(e.getMessage());
            e.printStackTrace();
        }
    }
}

