/*
Copyright (C) 2001, 2006 United States Government
as represented by the Administrator of the
National Aeronautics and Space Administration.
All Rights Reserved.
*/
package gov.nasa.worldwind.geom;

import java.awt.*;
import java.awt.geom.*;
import java.awt.image.*;
import java.util.*;

/**
 * @author tag
 * @version $Id: SurfacePolygon.java 1688 2007-05-02 22:03:39Z tgaskins $
 */
public class SurfacePolygon extends SurfaceShape
{

    public SurfacePolygon(Iterable<LatLon> positions, Color color, Color borderColor)
    {
        super(positions, color, borderColor);
    }

    public SurfacePolygon(Iterable<LatLon> positions)
    {
        super(positions, null, null);
    }

    protected final BufferedImage drawShape(BufferedImage image)
    {
        double minLat = this.getSector().getMinLatitude().getDegrees();
        double minLon = this.getSector().getMinLongitude().getDegrees();
        double dLat = this.getSector().getDeltaLatDegrees();
        double dLon = this.getSector().getDeltaLonDegrees();

        double latScale = dLat > 0 ? image.getHeight() / dLat : 0;
        double lonScale = dLon > 0 ? image.getWidth() / dLon : 0;

        GeneralPath path = new GeneralPath();

        Iterator<LatLon> positions = this.getPositions().iterator();
        if (!positions.hasNext())
            return image;

        LatLon pos = positions.next();
        path.moveTo(
            (float) Math.min(lonScale * (pos.getLongitude().getDegrees() - minLon), image.getWidth() - 1),
            (float) Math.min(latScale * (pos.getLatitude().getDegrees() - minLat), image.getHeight() - 1));

        double delta = 1d / this.getNumEdgeIntervals();
        while (positions.hasNext())
        {
            LatLon posNext = positions.next();
            for (int i = 0; i < this.getNumEdgeIntervals(); i++)
            {
                LatLon p = LatLon.interpolate(i * delta, pos, posNext);
                path.lineTo(
                    (float) Math.min(lonScale * (p.getLongitude().getDegrees() - minLon), image.getWidth() - 1),
                    (float) Math.min(latScale * (p.getLatitude().getDegrees() - minLat), image.getHeight() - 1));
            }

            // Set the last point directly to terminate any round-off error in the iteration above.
            path.lineTo(
                (float) Math.min(lonScale * (posNext.getLongitude().getDegrees() - minLon), image.getWidth() - 1),
                (float) Math.min(latScale * (posNext.getLatitude().getDegrees() - minLat), image.getHeight() - 1));

            pos = posNext;
        }

        Graphics2D g2 = image.createGraphics();

        if (this.isAntiAlias())
            g2.setRenderingHint(RenderingHints.KEY_ANTIALIASING, RenderingHints.VALUE_ANTIALIAS_ON);

        if (this.isDrawInterior())
        {
            g2.setPaint(this.getPaint());
            g2.fill(path);
        }

        if (this.isDrawBorder())
        {
            g2.setPaint(this.getBorderColor());
            g2.setStroke(this.getStroke());
            g2.draw(path);
        }

        return image;
    }
}
