/* Copyright (C) 2001, 2006 United States Government as represented by
the Administrator of the National Aeronautics and Space Administration.
All Rights Reserved.
*/
package worldwinddemo;

import gov.nasa.worldwind.*;
import gov.nasa.worldwind.awt.*;
import gov.nasa.worldwind.layers.*;
import gov.nasa.worldwind.layers.Earth.*;

import javax.swing.*;
import javax.swing.border.*;
import java.awt.*;
import java.awt.event.*;

/**
 * @author tag
 * @version $Id: WWPieceMaker.java 1764 2007-05-07 20:01:57Z tgaskins $
 */
public class BasicDemo
{
    private BasicDemo.LayerAction[] layers = new BasicDemo.LayerAction[] {
        new BasicDemo.LayerAction(new BMNGSurfaceLayer(), true),
        new BasicDemo.LayerAction(new LandsatI3(), true),
        new BasicDemo.LayerAction(new USGSDigitalOrtho(), false),
        new BasicDemo.LayerAction(new USGSUrbanAreaOrtho(), true),
        new BasicDemo.LayerAction(new EarthNASAPlaceNameLayer(), true),
        new BasicDemo.LayerAction(new CompassLayer(), true),
    };

    private static class AppFrame extends JFrame
    {
        private final WorldWindowGLCanvas wwd = new WorldWindowGLCanvas();

        public AppFrame(BasicDemo.LayerAction[] layers)
        {
            LayerList layerList = new LayerList();

            try
            {
                JPanel mainPanel = new JPanel();
                mainPanel.setLayout(new BorderLayout());
                wwd.setPreferredSize(new Dimension(800, 600));
                mainPanel.add(wwd, BorderLayout.CENTER);

                StatusBar statusBar = new StatusBar();
                statusBar.setEventSource(wwd);
                mainPanel.add(statusBar, BorderLayout.PAGE_END);
                this.getContentPane().add(mainPanel, BorderLayout.CENTER);

                JPanel westContainer = new JPanel(new BorderLayout());
                {
                    JPanel westPanel = new JPanel(new GridLayout(0, 1, 0, 10));
                    westPanel.setBorder(BorderFactory.createEmptyBorder(9, 9, 9, 9));
                    {
                        JPanel layersPanel = new JPanel(new GridLayout(0, 1, 0, 15));
                        layersPanel.setBorder(new TitledBorder("Layers"));
                        for (BasicDemo.LayerAction action : layers)
                        {
                            JCheckBox jcb = new JCheckBox(action);
                            jcb.setSelected(action.selected);
                            layersPanel.add(jcb);
                            layerList.add(action.layer);

                            if (action.layer instanceof TiledImageLayer)
                                ((TiledImageLayer) action.layer).setShowImageTileOutlines(false);

                            if (action.layer instanceof LandsatI3)
                                ((TiledImageLayer) action.layer).setDrawBoundingVolumes(false);

                            if (action.layer instanceof USGSDigitalOrtho)
                                ((TiledImageLayer) action.layer).setDrawTileIDs(false);
                        }
                        westPanel.add(layersPanel);
                        westContainer.add(westPanel, BorderLayout.NORTH);
                    }
                }

                this.getContentPane().add(westContainer, BorderLayout.WEST);
                this.pack();

                Dimension prefSize = this.getPreferredSize();
                prefSize.setSize(prefSize.getWidth(), 1.1 * prefSize.getHeight());
                this.setSize(prefSize);

                // Center the app on the user's screen.
                Dimension parentSize;
                Point parentLocation = new Point(0, 0);
                parentSize = Toolkit.getDefaultToolkit().getScreenSize();
                int x = parentLocation.x + (parentSize.width - prefSize.width) / 2;
                int y = parentLocation.y + (parentSize.height - prefSize.height) / 2;
                this.setLocation(x, y);
                this.setResizable(true);

                Model m = (Model) WorldWind.createConfigurationComponent(AVKey.MODEL_CLASS_NAME);
                m.setLayers(layerList);
                m.setShowWireframeExterior(false);
                m.setShowWireframeInterior(false);
                m.setShowTessellationBoundingVolumes(false);
                wwd.setModel(m);
            }
            catch (Exception e)
            {
                e.printStackTrace();
            }
        }
    }

    private static class LayerAction extends AbstractAction
    {
        private Layer layer;
        private boolean selected;

        public LayerAction(Layer layer, boolean selected)
        {
            super(layer.getName());
            this.layer = layer;
            this.selected = selected;
            this.layer.setEnabled(this.selected);
        }

        public void actionPerformed(ActionEvent actionEvent)
        {
            if (((JCheckBox) actionEvent.getSource()).isSelected())
                this.layer.setEnabled(true);
            else
                this.layer.setEnabled(false);

            appFrame.wwd.repaint();
        }
    }

    static
    {
        if (Configuration.isMacOS())
        {
            System.setProperty("apple.laf.useScreenMenuBar", "true");
            System.setProperty("com.apple.mrj.application.apple.menu.about.name", "World Wind Basic Demo");
            System.setProperty("com.apple.mrj.application.growbox.intrudes", "false");
        }
    }

    private static BasicDemo.AppFrame appFrame;

    public static void main(String[] args)
    {
        System.out.println("Java run-time version: " + System.getProperty("java.version"));
        System.out.println(gov.nasa.worldwind.Version.getVersion());

        try
        {
            BasicDemo demo = new BasicDemo();
            appFrame = new BasicDemo.AppFrame(demo.layers);
            appFrame.setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE);
            appFrame.setVisible(true);
        }
        catch (Exception e)
        {
            e.printStackTrace();
        }
    }
}
